
  /*
   *  Object %name    : %
   *  State           :  %state%
   *  Creation date   :  Mon Mar 28 12:09:38 2005
   *  Last modified   :  %modify_time%
   */
  /** @file
   *  \brief A brief description of this module
   *
   *  \version SST_TST_ATP_Data.c#1:csrc:1
   *  \author doronp
   *  \remarks Copyright (C) 2007 by Discretix Technologies Ltd.
   *           All Rights reserved
   */

/************* Include Files ***********************************/
#include "SST_TST.h"
#if TST_USE_DATA_MODULE
/************************ Defines ******************************/
/************************ Enums ********************************/
/************************ Typedefs *****************************/
/************************ Global Data **************************/
/************* Private function prototype **********************/
/************************ Public Functions *********************/
/*****************************************************************************
 * Function Name:                                                           
 *  SST_TST_ATP_BasicDataOp         
 * 
 * Inputs:
 *  None
 *
 * Outputs:
 *  DxError_t - Function error return                                        
 *
 * Description:                                                                 
 * Test the creation, deletion, modification, opening, closing and usage of a password authenticator object. 
 * And demonstrate the use of transaction with authenticator objects
 *  
 * Algorithm:       
 *    1.  Create a new session by calling: SST_SessionCreate.
 *    2.  Open a transaction by calling: SST_TransactionStart.
 *    3.  Create a new password authenticator by calling: SST_AuthenticatorPasswordCreate.
 *    4.  Close the transaction by calling: SST_TransactionEnd. 
 *    5.  Open the new authenticator by calling: SST_AuthenticationOpen.
 *    6.  Open a transaction by calling: SST_TransactionStart.
 *    7.  Insert a new data object by calling: SST_InsertData.
 *    8.  Close the transaction by calling: SST_TransactionEnd.
 *    9.  Read and compare the inserted data by calling: SST_ReadData.
 *    10.  Open a transaction by calling: SST_TransactionStart.
 *    11.  Modify the data from selected location by calling: SST_ModifyData.
 *    12.  Close the transaction by calling: SST_TransactionEnd.
 *    13.  Read and compare the modified data by calling: SST_ReadData.
 *    14.  Get the data size and compare to known by calling: SST_GetDataSize.
 *    15.  Open a transaction by calling: SST_TransactionStart.
 *    16.  Delete the data by calling: SST_DeleteData.
 *    17.  Delete the authenticator by calling: SST_AuthenticatorDelete.
 *    18.  Close the transaction by calling: SST_TransactionClose.
 *    19.  Delete the session by calling: SST_SessionDelete.
 *                            
 ********************************************************************************/    
DxError_t SST_TST_ATP_BasicDataOp ( void )
{
    /********* TST Variables Definitions ************/
    int                         TST_index,TST_HandleType;
    char                        TST_MasterTestName[TST_FUNC_STRING_LEN] = "SST_TST_ATP_BasicDataOp";
    SSTSessionId_t              TST_SessionId; 
    SSTTxnId_t                  TST_TransactionId;
    SSTHandle_t                 TST_AuthHandle,TST_DataHandle;
    SSTDataType_t               TST_SimpleData = TST_DATA_OBJECT;
    DxUint32_t                  TST_TempDataSize,TST_OutputDataSize;
    SSTHandle_t                 TST_AuthDefaultHandle;
    DxError_t                   TST_Status;

    /********* TST Variables initialization *********/

    ATP_LOG_TST_PRINT((SST_TST_MSG," ****************************************************** \n"));
    ATP_LOG_TST_PRINT((SST_TST_MSG," ** SST_TST_ATP_BasicDataOp Tests ********************* \n"));     
    ATP_LOG_TST_PRINT((SST_TST_MSG," ****************************************************** \n \n"));

    /************************************************/
    /*********** Start of test **********************/
    /************************************************/
    TST_Status = TESTPASS;

    for (TST_HandleType = TST_INTERNAL_HANDLE ; TST_HandleType < TST_HANDLES_TYPES ; TST_HandleType++)
    {
        for (TST_index =0 ; TST_index < TST_NumberOf_BasicDataOpObjectTests ; TST_index++) /*To run on all test vectors*/
        {
            if (TST_HandleType == TST_INTERNAL_HANDLE)
            {   
                ATP_LOG_TST_PRINT((SST_TST_MSG," ************************************ \n"));
                ATP_LOG_TST_PRINT((SST_TST_MSG," TEST using internal handles          \n"));
                ATP_LOG_TST_PRINT((SST_TST_MSG," ************************************ \n"));
                TST_AuthHandle.objId = SST_HANDLE_RANDOM_GENERATE; 
                TST_DataHandle.objId = SST_HANDLE_RANDOM_GENERATE;
            }
            else
            {
                ATP_LOG_TST_PRINT((SST_TST_MSG," ************************************ \n"));
                ATP_LOG_TST_PRINT((SST_TST_MSG," TEST using external handles          \n"));
                ATP_LOG_TST_PRINT((SST_TST_MSG," ************************************ \n"));
                TST_AuthHandle.objId = TST_TEST_DEFINED_HENDLE;
                TST_DataHandle.objId = TST_TEST_DEFINED_HENDLE_1;
            }

            ATP_LOG_TST_PRINT((SST_TST_MSG,"\n",TST_ATP_DataOpTestData[TST_index].TST_Name,"\n")); 

            /* Create a session for operations */
            /*---------------------------------*/
            TST_Status = SST_SessionCreate(&TST_SessionId);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_SessionCreate",
                          TST_ATP_DataOpTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Open a transaction for operations */
            /*-----------------------------------*/
            TST_Status = SST_TransactionStart(&TST_TransactionId);
      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"TST_TransactionId",
                          TST_ATP_DataOpTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Create a new password authenticator */
            /*-------------------------------------*/
            TST_AuthDefaultHandle.objDigest = 0;
      TST_AuthDefaultHandle.objId = SST_CLOSE_AUTH_HANDLE;
            TST_Status = SST_AuthenticatorPasswordCreate(TST_TransactionId,
                                                        TST_SessionId,
                                                        TST_ATP_DataOpTestData[TST_index].TST_Password,
                                                        TST_ATP_DataOpTestData[TST_index].TST_PasswordSizeInBytes,
                                                        TST_ATP_DataOpTestData[TST_index].TST_MaxNumOfTrials,
                                                        TST_AuthDefaultHandle,
                                                        &TST_AuthHandle,
                                                        TST_Workspace,
                                                        TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticatorPasswordCreate",
                          TST_ATP_DataOpTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Close the transaction */
            /*-----------------------*/
            TST_Status = SST_TransactionEnd(TST_TransactionId,
                       DX_TRUE,/*reflash sensitive*/
                       TST_Workspace,
                                           TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionEnd",
                          TST_ATP_DataOpTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Open the password authentication */
            /*----------------------------------*/
            TST_Status = SST_AuthenticationOpen(TST_SessionId,
                                               TST_AuthHandle,
                                               TST_ATP_DataOpTestData[TST_index].TST_Password,
                                               TST_ATP_DataOpTestData[TST_index].TST_PasswordSizeInBytes,
                                               TST_NO_CHALLENGE,
                                               TST_Workspace,
                                               TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticationOpen - Passeord",
                          TST_ATP_DataOpTestData[TST_index].TST_Name,
                          TST_MasterTestName);


      /* Open a transaction for operations */
            /*-----------------------------------*/
            TST_Status = SST_TransactionStart(&TST_TransactionId);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionStart",
                          TST_ATP_DataOpTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Insert a new Data object */
            /*--------------------------*/
            TST_Status = SST_DataInsert(TST_TransactionId,
                                       TST_SessionId, 
                                       TST_AuthHandle,       
                                       TST_ATP_DataOpTestData[TST_index].TST_Data,
                                       TST_ATP_DataOpTestData[TST_index].TST_DataSize,
                                       DX_TRUE,
                                       TST_SimpleData,                
                                       &TST_DataHandle,
                                       TST_Workspace,
                                       TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_DataInsert",
                          TST_ATP_DataOpTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Close the transaction */
            /*-----------------------*/
            TST_Status = SST_TransactionEnd(TST_TransactionId,
                       DX_TRUE,/*reflash sensitive*/
                       TST_Workspace,
                                           TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionEnd",
                          TST_ATP_DataOpTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Read the data */
            /*---------------*/
            TST_TempDataSize = TST_ATP_DataOpTestData[TST_index].TST_DataSize;

            TST_Status = SST_DataRead(TST_SessionId,
                                     TST_DataHandle,
                                     TST_ATP_DataOpTestData[TST_index].TST_DataReadOffset,
                                     TST_ATP_DataOpTestData[TST_index].TST_OutputData,
                                     &TST_TempDataSize,
                                     TST_Workspace,
                                     TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_DataRead",
                          TST_ATP_DataOpTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Compare the retrieved data to the original */
      /*--------------------------------------------*/
         if(!DX_VOS_MemCmp(TST_ATP_DataOpTestData[TST_index].TST_Data,
                TST_ATP_DataOpTestData[TST_index].TST_OutputData,
                TST_ATP_DataOpTestData[TST_index].TST_DataSize))
                              TST_Status = TESTPASS; /*Data Buffers are the same*/

                              else
                              TST_Status = TESTFAIL; /*Data Buffers are different*/
        
      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"DX_VOS_MemCmp - Original data",
                          TST_ATP_DataOpTestData[TST_index].TST_Name,
                          TST_MasterTestName);


  

            /* Open a transaction for operations */
            /*-----------------------------------*/
            TST_Status = SST_TransactionStart(&TST_TransactionId);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionStart",
                          TST_ATP_DataOpTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Modify the data from selected location */
            /*----------------------------------------*/
            TST_Status = SST_DataModify(TST_TransactionId,
                                       TST_SessionId,                           
                                       TST_DataHandle,  
                                       TST_ATP_DataOpTestData[TST_index].TST_DataToWrite,
                                       TST_ATP_DataOpTestData[TST_index].TST_DataToWriteSize,
                                       TST_ATP_DataOpTestData[TST_index].TST_DataToWriteOffset,
                                       TST_Workspace,
                                       TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_DataModify",
                          TST_ATP_DataOpTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Close the transaction */
            /*-----------------------*/
            TST_Status = SST_TransactionEnd(TST_TransactionId,
                       DX_TRUE,/*reflash sensitive*/
                       TST_Workspace,
                                           TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionEnd",
                          TST_ATP_DataOpTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Read the data */
            /*---------------*/
            TST_TempDataSize = TST_ATP_DataOpTestData[TST_index].TST_DataSize;

            TST_Status = SST_DataRead(TST_SessionId,
                                     TST_DataHandle,
                                     TST_ATP_DataOpTestData[TST_index].TST_DataReadOffset,
                                     TST_ATP_DataOpTestData[TST_index].TST_OutputData,
                                     &TST_TempDataSize,
                                     TST_Workspace,
                                     TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_DataRead",
                          TST_ATP_DataOpTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Compare the retrieved data to the original */
      /*--------------------------------------------*/
         if(!DX_VOS_MemCmp(TST_ATP_DataOpTestData[TST_index].TST_ModifedData,
                TST_ATP_DataOpTestData[TST_index].TST_OutputData,
                TST_ATP_DataOpTestData[TST_index].TST_DataSize))
                              TST_Status = TESTPASS; /*Data Buffers are the same*/
                              else
                              TST_Status = TESTFAIL; /*Data Buffers are different*/

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"DX_VOS_MemCmp - Modified data",
                          TST_ATP_DataOpTestData[TST_index].TST_Name,
                          TST_MasterTestName);



            /* Get Data Size and compare */
            /*---------------------------*/
      TST_Status = SST_DataSizeGet(TST_SessionId,  
                                        TST_DataHandle,
                                        &TST_OutputDataSize,
                                        TST_Workspace,
                                        TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_DataSizeGet",
                          TST_ATP_DataOpTestData[TST_index].TST_Name,
                          TST_MasterTestName);

             
            /* Compare retrieved data size to original */
            /*-----------------------------------------*/
         if(!DX_VOS_MemCmp(&TST_OutputDataSize,
                &TST_ATP_DataOpTestData[TST_index].TST_DataSize,
                sizeof(TST_OutputDataSize)))
                              TST_Status = TESTPASS; /*Data Buffers are the same*/
                              else
                              TST_Status = TESTFAIL; /*Data Buffers are different*/

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"DX_VOS_MemCmp - data size",
                          TST_ATP_DataOpTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Open a transaction for operations */
            /*-----------------------------------*/
            TST_Status = SST_TransactionStart(&TST_TransactionId);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionStart",
                          TST_ATP_DataOpTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Delete the data */
            /*-----------------*/
             TST_Status = SST_DataDelete(TST_TransactionId,
                                        TST_SessionId,  
                                        TST_DataHandle,
                                        TST_Workspace,
                                        TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_DataDelete",
                          TST_ATP_DataOpTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Delete the authenticator */
            /*--------------------------*/
            TST_Status = SST_AuthenticatorDelete(TST_TransactionId,
                                                TST_AuthHandle);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticatorDelete - Master authenticator",
                          TST_ATP_DataOpTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Close the transaction */
            /*-----------------------*/
            TST_Status = SST_TransactionEnd(TST_TransactionId,
                       DX_TRUE,/*reflash sensitive*/
                       TST_Workspace,
                                           TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionEnd",
                          TST_ATP_DataOpTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Delete the session */
            /*--------------------*/
            TST_Status = SST_SessionDelete(TST_SessionId);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_SessionDelete",
                          TST_ATP_DataOpTestData[TST_index].TST_Name,
                          TST_MasterTestName);


        }/*End of TST_index "for loop" */
    }/*End of TST_HandleType "for loop" */
    return TST_Status;

  EXIT_ON_ERROR:
  {
    return DX_FAILURE;
  }
}/* End of SST_TST_ATP_BasicDataOp */

/***************************************************************************************************************************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***************************************************************************************************************************************************************/


/***************************************************************************************************************************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***************************************************************************************************************************************************************/
/*****************************************************************************
 * Function Name:                                                           
 *  SST_TST_ATP_BindingWritingPermissions         
 * 
 * Inputs:
 *  None
 *
 * Outputs:
 *  DxError_t - Function error return                                        
 *
 * Description:                                                                 
 * Test the bind and unbind authenticator to data object capabilities of the SST
 * NOTE: this is a test that has error return values as a correct matter or operation!
 *  
 * Algorithm:       
 *    1.  Build a Private Key using CRYS for CRYS usage by calling: CRYS_RSA_Build_PrivKey (externally generated key).
 *    2.  Create a new session  session#1 by calling: SST_SessionCreate.
 *    3.  Create a new session  session#2 by calling: SST_SessionCreate.
 *    4.  Create a new session  session#3 by calling: SST_SessionCreate.
 *    5.  Open a transaction by calling: SST_TransactionStart.
 *    6.  Create a new password authenticator by calling: SST_AuthenticatorPasswordCreate.
 *    7.  Create a new asymmetric key authenticator by calling: SST_AuthenticatorPubKeyRSACreate. 
 *    8.  Create a new asymmetric key authenticator by calling: SST_AuthenticatorAESSharedSecretCreate.
 *    9.  Open the public key authenticator with session id#1 by calling:
 *            i.      SST_AuthenticatorChallengeGet 
 *            ii.  CRYS_RSA_PSS_Sign
 *            iii.  SST_AuthenticationOpen.
 *    10.  Close the transaction by calling: SST_TransactionClose.
 *    11.  Open a transaction by calling: SST_TransactionStart.
 *    12.  Open the AES shared key authenticator with session id#1 by calling:
 *            i.      SST_AuthenticatorChallengeGet 
 *            ii.  CRYS_AES (Encrypt the challenge using the AES key)
 *            iii.  SST_AuthenticationOpen.
 *    13.  Close the transaction by calling: SST_TransactionClose.
 *    14.  Open a transaction by calling: SST_TransactionStart.
 *    15.  Insert a new data object by calling: SST_InsertData with the Public key authenticator as master authenticator with session id#1.
 *    16.  Bind the password authenticator to the data object with read only permissions by calling: SST_BindAuthenticatorToElement using session id#1.
 *    17.  Bind the AES shared key authenticator to the data object with read/write permissions by calling: SST_BindAuthenticatorToElement using session id#1.
 *    18.  Close all the authenticators (one by one) by calling: SST_AuthenticationClose.
 *    19.  Close the transaction by calling: SST_TransactionClose.
 *    20.  Open the password authenticator by calling: SST_AuthenticationOpen. using session id#2
 *    21.  Open a transaction by calling: SST_TransactionStart.
 *    22.  Using session id#2 read and compare the data by calling: SST_ReadData. 
 *    23.  Using session id#2 try to modify the data by calling: SST_ModifyData. (This should fail!)
 *    24.  Try to bind the AES shared key authenticator to the data by calling: SST_BindAuthenticatorToElement. (This should fail!).
 *    25.  Close the authenticator by calling: SST_AuthenticationClose.
 *    26.  Using session #3 open the AES shared key authenticator by calling:
 *            i.      SST_AuthenticatorChallengeGet 
 *            ii.  CRYS_AES (Encrypt the challenge using the AES key)
 *            iii.  SST_AuthenticationOpen.
 *    27.  Close the transaction by calling: SST_TransactionClose.
 *    28.  Open a transaction by calling: SST_TransactionStart.
 *    29.  Modify the data from selected location by calling: SST_ModifyData.
 *    30.  Close the transaction by calling: SST_TransactionClose.
 *    31.  Using session #3 open the password authenticator by calling: SST_AuthenticationOpen.
 *    32.  Open a transaction by calling: SST_TransactionStart.
 *    33.  Using session #3 bind the password authenticator to the data by calling: SST_BindAuthenticatorToElement. (This should fail!).
 *    34.  Using session #3 read and compare the modified data by calling: SST_ReadData.
 *    35.  Using session #3 get the data size and compare to known by calling: SST_GetDataSize.
 *    36.  Open the public key authenticator with session id#1 by calling:
 *            i.      SST_AuthenticatorChallengeGet 
 *            ii.  CRYS_RSA_PSS_Sign
 *            iii.  SST_AuthenticationOpen.
 *    37.  Close the transaction by calling: SST_TransactionClose.
 *    38.  Open a transaction by calling: SST_TransactionStart.
 *    39.  Unbind the data from all other authenticators by calling: SST_AuthenticatorUnbind
 *    40.  Using session #1 and the master authenticator delete the data by calling: SST_DeleteData.
 *    41.  Close all of the authenticators (one at a time) by calling: SST_AuthenticationClose.
 *    42.  Delete all of the authenticators (one at a time) by calling: SST_AuthenticatorDelete.
 *    43.  Close the transaction by calling: SST_TransactionClose.
 *    44.  Delete all of the sessions (one at a time) by calling: SST_SessionDelete.
 *                         
 *******************************************************************************/    
DxError_t SST_TST_ATP_BindingWritingPermissions ( void )
{
    /********* TST Variables Definitions ************/
    int                                TST_index,TST_HandleType;
    char                               TST_MasterTestName[TST_FUNC_STRING_LEN] = "SST_TST_ATP_BindingWritingPermissions";
    SSTSessionId_t                     TST_SessionId1,TST_SessionId2,TST_SessionId3; 
    SSTTxnId_t                         TST_TransactionId;
    SSTHandle_t                        TST_AuthHandlePwd,TST_AuthHandleAES,TST_AuthHandlePub;
    SSTHandle_t                        TST_DataHandle;
    SSTDataType_t                      TST_SimpleData = TST_DATA_OBJECT;
    DxUint32_t                         TST_TempDataSize,TST_OutputDataSize;
    SSTChallengeId_t                   TST_ChallengeId;
    static CRYS_RSAUserPrivKey_t       TST_CRYS_RSAUserPrivKey;
    static CRYS_RSAPrivUserContext_t   TST_CRYS_RSAPrivUserContext;
  SSTHandle_t TST_AuthDefaultHandle;
  DxError_t                 TST_Status;
    /********* TST Variables initialization *********/

    ATP_LOG_TST_PRINT((SST_TST_MSG," ****************************************************** \n"));
    ATP_LOG_TST_PRINT((SST_TST_MSG," ** SST_TST_ATP_BindingWritingPermissions Tests ******* \n"));     
    ATP_LOG_TST_PRINT((SST_TST_MSG," ****************************************************** \n \n"));

    /************************************************/
    /*********** Start of test **********************/
    /************************************************/
    for (TST_HandleType = TST_INTERNAL_HANDLE ; TST_HandleType < TST_HANDLES_TYPES ; TST_HandleType++)
    {
        for (TST_index =0 ; TST_index < TST_NumberOf_BindingWritingPermissionsObjectTests ; TST_index++) /*To run on all test vectors*/
        {
            if (TST_HandleType == TST_INTERNAL_HANDLE)
            {   
                ATP_LOG_TST_PRINT((SST_TST_MSG," ************************************ \n"));
                ATP_LOG_TST_PRINT((SST_TST_MSG," TEST using internal handles          \n"));
                ATP_LOG_TST_PRINT((SST_TST_MSG," ************************************ \n"));
                TST_AuthHandlePwd.objId = SST_HANDLE_RANDOM_GENERATE;
                TST_AuthHandleAES.objId = SST_HANDLE_RANDOM_GENERATE;
                TST_AuthHandlePub.objId = SST_HANDLE_RANDOM_GENERATE;
                TST_DataHandle.objId = SST_HANDLE_RANDOM_GENERATE;
            }
            else
            {
                ATP_LOG_TST_PRINT((SST_TST_MSG," ************************************ \n"));
                ATP_LOG_TST_PRINT((SST_TST_MSG," TEST using external handles          \n"));
                ATP_LOG_TST_PRINT((SST_TST_MSG," ************************************ \n"));
                TST_AuthHandlePwd.objId = TST_TEST_DEFINED_HENDLE;
                TST_AuthHandleAES.objId = TST_TEST_DEFINED_HENDLE;
                TST_AuthHandlePub.objId = TST_TEST_DEFINED_HENDLE;
                TST_DataHandle.objId = SST_HANDLE_RANDOM_GENERATE;
            }

            ATP_LOG_TST_PRINT((SST_TST_MSG,"\n",TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,"\n")); 

            /* Build the Private Key for CRYS usage */
            /*--------------------------------------*/
            TST_Status = CRYS_RSA_Build_PrivKey(&TST_CRYS_RSAUserPrivKey,
                                               TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_D,
                                               (DxUint16_t)TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_DSizeInBytes,
                                               TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_E,
                                               (DxUint16_t)TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_ESizeInBytes,
                                               TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_N,
                                               (DxUint16_t)TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_NSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_Build_PrivKey",
                    TST_ATP_RsaAuthObjectTestData[TST_index].TST_Name,
                    TST_MasterTestName);


            /* Create a session #1 for operations */
            /*------------------------------------*/
            TST_Status = SST_SessionCreate(&TST_SessionId1);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_SessionCreate",
                          TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Create a session #2 for operations */
            /*------------------------------------*/
            TST_Status = SST_SessionCreate(&TST_SessionId2);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_SessionCreate",
                          TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Create a session #3 for operations */
            /*------------------------------------*/
            TST_Status = SST_SessionCreate(&TST_SessionId3);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_SessionCreate",
                          TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Open a transaction for operations */
            /*-----------------------------------*/
            TST_Status = SST_TransactionStart(&TST_TransactionId);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionStart",
                          TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Create a new password authenticator using session 1 */
            /*-----------------------------------------------------*/
            TST_AuthDefaultHandle.objDigest = 0;
      TST_AuthDefaultHandle.objId = SST_CLOSE_AUTH_HANDLE;
            TST_Status = SST_AuthenticatorPasswordCreate(TST_TransactionId,
                                                        TST_SessionId1,
                                                        TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Password,
                                                        TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_PasswordSizeInBytes,
                                                        TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_MaxNumOfTrials,
                                                        TST_AuthDefaultHandle,
                                                        &TST_AuthHandlePwd,
                                                        TST_Workspace,
                                                        TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticatorPasswordCreate",
                          TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Create a new symmetric key authenticator */
            /*------------------------------------------*/
            TST_Status = SST_AuthenticatorAESSharedSecretCreate(TST_TransactionId,
                                                               TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_AesKey,
                                                               SST_KEY_TYPE_AES_128_BIT,
                                                               &TST_AuthHandleAES,
                                                               TST_Workspace,
                                                               TST_WorkspaceSizeInBytes);

            TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticatorAESSharedSecretCreate",
                          TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Create a new public key authenticator */
            /*---------------------------------------*/
            TST_Status = SST_AuthenticatorPubKeyRSACreate(TST_TransactionId,
                                                      TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_N,
                                                      TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_E,
                                                      TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_ESizeInBytes,
                                                      &TST_AuthHandlePub,
                                                      TST_Workspace,
                                                      TST_WorkspaceSizeInBytes);

            TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticatorPubKeyRSACreate",
                          TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Open the public key authentication using session 1 */
            /*----------------------------------------------------*/
            /* Get the challenge*/
            TST_Status = SST_AuthenticatorChallengeGet(TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_ChallengeBuff,
                                                      TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_ChallengeBuffSize,  
                                                      &TST_ChallengeId);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticatorChallengeGet",
                          TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Sign the Challenge buffer */
            TST_Status = CRYS_RSA_PSS_Sign(&TST_CRYS_RSAPrivUserContext,
                                               &TST_CRYS_RSAUserPrivKey,
                                               CRYS_RSA_HASH_SHA1_mode,
                         CRYS_PKCS1_MGF1,
                         20,
                                               TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_ChallengeBuff,
                                               TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_ChallengeBuffSize,
                                               TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_SignedChallengeBuff,
                                               (DxUint16_t*)&TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_SignedChallengeBuffSize);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_PSS_Sign",
                          TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                          TST_MasterTestName);



            /* Close the transaction */
            /*-----------------------*/
            TST_Status = SST_TransactionEnd(TST_TransactionId,
                       DX_TRUE,/*reflash sensitive*/
                       TST_Workspace,
                                           TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionEnd",
                          TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Open the authenticator */
            TST_Status = SST_AuthenticationOpen(TST_SessionId1,
                                               TST_AuthHandlePub,
                                               TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_SignedChallengeBuff,
                                               TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_SignedChallengeBuffSize,
                                               TST_ChallengeId,
                                               TST_Workspace,
                                               TST_WorkspaceSizeInBytes);                                                              

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticationOpen - Public",
                          TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                          TST_MasterTestName);


      /* Open a transaction for operations */
            /*-----------------------------------*/
            TST_Status = SST_TransactionStart(&TST_TransactionId);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionStart",
                          TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Open the AES key authentication using session 1 */
            /*-------------------------------------------------*/
            /* Get the challenge*/
            TST_Status = SST_AuthenticatorChallengeGet(TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_ChallengeBuff,
                                                      TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_ChallengeBuffSize,  
                                                      &TST_ChallengeId);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticatorChallengeGet",
                          TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Encrypt the challenge using the AES key */
            TST_Status = CRYS_AES(DX_NULL,
                                 TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_AesKey,
                                 CRYS_AES_Key128BitSize,
                                 CRYS_AES_Encrypt,
                                 CRYS_AES_ECB_mode,
                                 TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_ChallengeBuff,
                                 TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_ChallengeBuffSize,
                                 TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_EncChallengeBuff);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_AES",
                          TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                          TST_MasterTestName);

            
      /* Close the transaction */
            /*-----------------------*/
            TST_Status = SST_TransactionEnd(TST_TransactionId,
                       DX_TRUE,/*reflash sensitive*/
                       TST_Workspace,
                                           TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionEnd",
                          TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Open the authenticator */
            TST_Status = SST_AuthenticationOpen(TST_SessionId1,
                                               TST_AuthHandleAES,
                                               TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_EncChallengeBuff,
                                               TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_ChallengeBuffSize,
                                               TST_ChallengeId,
                                               TST_Workspace,
                                               TST_WorkspaceSizeInBytes);               
                                               
      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticationOpen-AES",
                          TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Open the password authentication using session 1 */
            /*--------------------------------------------------*/
            TST_Status = SST_AuthenticationOpen(TST_SessionId1,
                                               TST_AuthHandlePwd,
                                               TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Password,
                                               TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_PasswordSizeInBytes,
                                               TST_NO_CHALLENGE,
                                               TST_Workspace,
                                               TST_WorkspaceSizeInBytes);               
                                               
      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticationOpen - Master Authentication",
                          TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                          TST_MasterTestName);


      /* Open a transaction for operations */
            /*-----------------------------------*/
            TST_Status = SST_TransactionStart(&TST_TransactionId);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"TST_TransactionId",
                          TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                          TST_MasterTestName);
  

            /* Insert a new Data object with pub key authentication and session 1 */
            /*--------------------------------------------------------------------*/
            TST_Status = SST_DataInsert(TST_TransactionId,
                                       TST_SessionId1, 
                                       TST_AuthHandlePub,       
                                       TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Data,
                                       TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_DataSize,
                                       DX_TRUE,
                                       TST_SimpleData,                
                                       &TST_DataHandle,
                                       TST_Workspace,
                                       TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_DataInsert",
                          TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                          TST_MasterTestName);
  

            /*Bind the data object to password authentication with read only permissions */
            /*---------------------------------------------------------------------------*/
            TST_Status = SST_AuthenticatorBind(TST_TransactionId,  
                                              TST_SessionId1, 
                                              TST_DataHandle,              
                                              TST_AuthHandlePwd,             
                                              SST_PERMISSION_READ_ONLY,
                                              TST_Workspace,
                                              TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticatorBind - password auth/READ_ONLY",
                          TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /*Bind the data object to AES key authentication with modify permissions */
            /*-----------------------------------------------------------------------*/
            TST_Status = SST_AuthenticatorBind(TST_TransactionId,  
                                              TST_SessionId1, 
                                              TST_DataHandle,              
                                              TST_AuthHandleAES,             
                                              SST_PERMISSION_MODIFY,
                                              TST_Workspace,
                                              TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticatorBind - password auth/MODIFY",
                          TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Close all authenticators */
            /*--------------------------*/
            /* AES shared secret*/
            TST_Status = SST_AuthenticationClose(TST_SessionId1,TST_AuthHandleAES);
            
      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticationClose-TST_AuthHandleAES",
                          TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /*Password*/
            TST_Status = SST_AuthenticationClose(TST_SessionId1,TST_AuthHandlePwd);
            
      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticationClose-TST_AuthHandlePwd",
                          TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Public key*/
            TST_Status = SST_AuthenticationClose(TST_SessionId1,TST_AuthHandlePub);
            
      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticationClose-TST_AuthHandlePub",
                          TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Close the transaction */
            /*-----------------------*/
            TST_Status = SST_TransactionEnd(TST_TransactionId,
                       DX_TRUE,/*reflash sensitive*/
                       TST_Workspace,
                                           TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionEnd",
                          TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Open the password authentication using session 2 */
            /*--------------------------------------------------*/
            TST_Status = SST_AuthenticationOpen(TST_SessionId2,
                                               TST_AuthHandlePwd,
                                               TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Password,
                                               TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_PasswordSizeInBytes,
                                               TST_NO_CHALLENGE,
                                               TST_Workspace,
                                               TST_WorkspaceSizeInBytes);               
                                               
      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticationOpen - Master Authentication",
                          TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                          TST_MasterTestName);


      /* Open a transaction for operations */
            /*-----------------------------------*/
            TST_Status = SST_TransactionStart(&TST_TransactionId);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionStart",
                          TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Read and compare the data using session 2 */
            /*-------------------------------------------*/
            TST_TempDataSize = TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_DataSize;

            TST_Status = SST_DataRead(TST_SessionId2,
                                     TST_DataHandle,
                                     TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_DataReadOffset,
                                     TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_OutputData,
                                     &TST_TempDataSize,
                                     TST_Workspace,
                                     TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_DataRead",
                          TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Compare the retrieved data to the original */
         if(!DX_VOS_MemCmp(TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Data,
                TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_OutputData,
                TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_DataSize))
                              TST_Status = TESTPASS; /*Data Buffers are the same*/
                              else
                              TST_Status = TESTFAIL; /*Data Buffers are different*/

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"DX_VOS_MemCmp - Original data",
                          TST_ATP_DataOpTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Try to modify the data using session 2 */
            /* !!! THIS SHOULD FAIL !!!               */
            /*----------------------------------------*/
            TST_Status = SST_DataModify(TST_TransactionId,
                                       TST_SessionId2,                           
                                       TST_DataHandle,   
                                       TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Data,
                                       TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_DataSize,
                                       TST_NO_OFFSET,
                                       TST_Workspace,
                                       TST_WorkspaceSizeInBytes);

            TST_Status = TST_CheckExpectedErrorReturn(TST_Status, 
                                                     SST_RC_ERROR_ACCESS_DENIED, 
                                                     "SST_DataModify",
                                                     TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                                                     TST_MasterTestName);

      #if SST_TST_EXIT_ON_ERROR
        if (TST_Status != TESTPASS)
          goto EXIT_ON_ERROR;
      #endif

            /* Bind the AES shared authenticator to the data */
            /* !!! THIS SHOULD FAIL!!!!                      */
            /*-----------------------------------------------*/
            TST_Status = SST_AuthenticatorBind(TST_TransactionId,  
                                              TST_SessionId2, 
                                              TST_DataHandle,              
                                              TST_AuthHandleAES,             
                                              SST_PERMISSION_MODIFY,
                                              TST_Workspace,
                                              TST_WorkspaceSizeInBytes);

            TST_Status = TST_CheckExpectedErrorReturn(TST_Status, 
                                                     SST_RC_ERROR_ACCESS_DENIED, 
                                                     "SST_AuthenticatorBind",
                                                     TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                                                     TST_MasterTestName);

      #if SST_TST_EXIT_ON_ERROR
        if (TST_Status != TESTPASS)
        goto EXIT_ON_ERROR;

      #endif

            /* Close the password authentication */
            /*-----------------------------------*/
            TST_Status = SST_AuthenticationClose(TST_SessionId2,TST_AuthHandlePwd);
            
      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticationClose-TST_AuthHandlePwd",
                          TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Open the AES key authentication using session 3 */
            /*-------------------------------------------------*/
            /* Get the challenge*/
            TST_Status = SST_AuthenticatorChallengeGet(TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_ChallengeBuff,
                                                      TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_ChallengeBuffSize,  
                                                      &TST_ChallengeId);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticatorChallengeGet",
                          TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                          TST_MasterTestName);
  

            /* Encrypt the challenge using the AES key */
            TST_Status = CRYS_AES(DX_NULL,
                                 TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_AesKey,
                                 CRYS_AES_Key128BitSize,
                                 CRYS_AES_Encrypt,
                                 CRYS_AES_ECB_mode,
                                 TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_ChallengeBuff,
                                 TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_ChallengeBuffSize,
                                 TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_EncChallengeBuff);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_AES",
                          TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                          TST_MasterTestName);


      /* Close the transaction */
            /*-----------------------*/
            TST_Status = SST_TransactionEnd(TST_TransactionId,
                       DX_TRUE,/*reflash sensitive*/
                       TST_Workspace,
                                           TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionEnd",
                          TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Open the authenticator */
            TST_Status = SST_AuthenticationOpen(TST_SessionId3,
                                               TST_AuthHandleAES,
                                               TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_EncChallengeBuff,
                                               TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_ChallengeBuffSize,
                                               TST_ChallengeId,
                                               TST_Workspace,
                                               TST_WorkspaceSizeInBytes);               
                                               
      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticationOpen-AES",
                          TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                          TST_MasterTestName);

      
      /* Open a transaction for operations */
            /*-----------------------------------*/
            TST_Status = SST_TransactionStart(&TST_TransactionId);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"TST_TransactionId",
                          TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                          TST_MasterTestName);

            /* Try to modify the data using session 3 */
            /*----------------------------------------*/
            TST_Status = SST_DataModify(TST_TransactionId,
                                       TST_SessionId3,                           
                                       TST_DataHandle,   
                                       TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Data,
                                       TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_DataSize,
                                       TST_NO_OFFSET,
                                       TST_Workspace,
                                       TST_WorkspaceSizeInBytes);

            TST_ATP_CHECK_ERROR_RETURN(TST_Status, 
                                             "SST_DataModify",
                                             TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                                             TST_MasterTestName);



      /* Close the transaction */
            /*-----------------------*/
            TST_Status = SST_TransactionEnd(TST_TransactionId,
                       DX_TRUE,/*reflash sensitive*/
                       TST_Workspace,
                                           TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionEnd",
                          TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Open the password authentication using session 2 */
            /*--------------------------------------------------*/
            TST_Status = SST_AuthenticationOpen(TST_SessionId2,
                                               TST_AuthHandlePwd,
                                               TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Password,
                                               TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_PasswordSizeInBytes,
                                               TST_NO_CHALLENGE,
                                               TST_Workspace,
                                               TST_WorkspaceSizeInBytes);               
                                               
      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticationOpen - Password",
                          TST_ATP_DataOpTestData[TST_index].TST_Name,
                          TST_MasterTestName);


      /* Open a transaction for operations */
            /*-----------------------------------*/
            TST_Status = SST_TransactionStart(&TST_TransactionId);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionStart",
                          TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Bind the password authenticator to the data */
            /* !!! THIS SHOULD FAIL!!!!                    */
            /*---------------------------------------------*/
             TST_Status= SST_AuthenticatorBind(TST_TransactionId,  
                                              TST_SessionId3, 
                                              TST_DataHandle,              
                                              TST_AuthHandleAES,             
                                              SST_PERMISSION_MODIFY,
                                              TST_Workspace,
                                              TST_WorkspaceSizeInBytes);

            TST_Status = TST_CheckExpectedErrorReturn(TST_Status, 
                                                     SST_RC_ERROR_ACCESS_DENIED, 
                                                     "SST_AuthenticatorBind",
                                                     TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                                                     TST_MasterTestName);

      #if SST_TST_EXIT_ON_ERROR
        if (TST_Status != TESTPASS)
          goto EXIT_ON_ERROR;
      #endif

            /* Read and compare the data using session 2 */
            /*-------------------------------------------*/
            TST_TempDataSize = TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_DataSize;

            TST_Status = SST_DataRead(TST_SessionId2,
                                     TST_DataHandle,
                                     TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_DataReadOffset,
                                     TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_OutputData,
                                     &TST_TempDataSize,
                                     TST_Workspace,
                                     TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_DataRead",
                          TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Compare the retrieved data to the original */
         if(!DX_VOS_MemCmp(TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Data,
                TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_OutputData,
                TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_DataSize))
                              TST_Status = TESTPASS; /*Data Buffers are the same*/
                              else
                              TST_Status = TESTFAIL; /*Data Buffers are different*/

            TST_ATP_CHECK_ERROR_RETURN(TST_Status,"DX_VOS_MemCmp - Original data",
                          TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Get Data Size and compare */
            /*---------------------------*/
      TST_Status = SST_DataSizeGet(TST_SessionId2,  
                                        TST_DataHandle,
                                        &TST_OutputDataSize,
                                        TST_Workspace,
                                        TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_DataSizeGet",
                          TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Open the public key authentication using session 1 */
            /*----------------------------------------------------*/
            /* Get the challenge*/
            TST_Status = SST_AuthenticatorChallengeGet(TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_ChallengeBuff,
                                                      TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_ChallengeBuffSize,  
                                                      &TST_ChallengeId);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticatorChallengeGet",
                          TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Sign the Challenge buffer */
           TST_Status = CRYS_RSA_PSS_Sign(&TST_CRYS_RSAPrivUserContext,
                                               &TST_CRYS_RSAUserPrivKey,
                                               CRYS_RSA_HASH_SHA1_mode,
                         CRYS_PKCS1_MGF1,
                         20,
                                               TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_ChallengeBuff,
                                               TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_ChallengeBuffSize,
                                               TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_SignedChallengeBuff,
                                               (DxUint16_t*)&TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_SignedChallengeBuffSize);
                                               
      
      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"CRYS_RSA_PSS_Sign",
                          TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                          TST_MasterTestName);


      /* Close the transaction */
            /*-----------------------*/
            TST_Status = SST_TransactionEnd(TST_TransactionId,
                       DX_TRUE,/*reflash sensitive*/
                       TST_Workspace,
                                           TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionEnd",
                          TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Open the authenticator */
            TST_Status = SST_AuthenticationOpen(TST_SessionId1,
                                               TST_AuthHandlePub,
                                               TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_SignedChallengeBuff,
                                               TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_SignedChallengeBuffSize,
                                               TST_ChallengeId,
                                               TST_Workspace,
                                               TST_WorkspaceSizeInBytes);               
                                               

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticationOpen - Public Key",
                          TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                          TST_MasterTestName);


      /* Open a transaction for operations */
            /*-----------------------------------*/
            TST_Status = SST_TransactionStart(&TST_TransactionId);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionStart",
                          TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Unbind the data from all other authenticators */
            /*-----------------------------------------------*/
            /* Unbind from AES authenticator*/
            TST_Status = SST_AuthenticatorUnbind(TST_TransactionId,   
                                                TST_SessionId1,
                                                TST_DataHandle,
                                                TST_AuthHandleAES,              
                                                SST_PERMISSION_MODIFY,
                                                TST_Workspace,
                                                TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticatorUnbind - AES Authenticator",
                          TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Unbind from password authenticator*/
            TST_Status = SST_AuthenticatorUnbind(TST_TransactionId,   
                                                TST_SessionId1,
                                                TST_DataHandle,
                                                TST_AuthHandlePwd,              
                                                SST_PERMISSION_READ_ONLY,
                                                TST_Workspace,
                                                TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticatorUnbind - Password Authenticator",
                          TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Delete the data using session 1 */
            /*---------------------------------*/
             TST_Status = SST_DataDelete(TST_TransactionId,
                                        TST_SessionId1,  
                                        TST_DataHandle,
                                        TST_Workspace,
                                        TST_WorkspaceSizeInBytes);


      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_DataDelete",
                          TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Close all authenticators */
            /*--------------------------*/
            /* AES shared secret*/
            TST_Status = SST_AuthenticationClose(TST_SessionId3,TST_AuthHandleAES);
            
      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticationClose-TST_AuthHandleAES",
                          TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /*Password on session 2*/
            TST_Status = SST_AuthenticationClose(TST_SessionId2,TST_AuthHandlePwd);
            
      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticationClose-TST_AuthHandlePwd - session2",
                          TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Public key*/
            TST_Status = SST_AuthenticationClose(TST_SessionId1,TST_AuthHandlePub);
            
      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticationClose-TST_AuthHandlePub",
                          TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Delete the authenticators */
            /*---------------------------*/
            /* Password */
            TST_Status = SST_AuthenticatorDelete(TST_TransactionId,
                                                TST_AuthHandlePwd);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticatorDelete - password",
                          TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* AES shared secret */
            TST_Status = SST_AuthenticatorDelete(TST_TransactionId,
                                                TST_AuthHandleAES);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticatorDelete - AES",
                          TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Public key */
            TST_Status = SST_AuthenticatorDelete(TST_TransactionId,
                                                TST_AuthHandlePub);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticatorDelete - Public key",
                          TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Close the transaction */
            /*-----------------------*/
            TST_Status = SST_TransactionEnd(TST_TransactionId,
                       DX_TRUE,/*reflash sensitive*/
                       TST_Workspace,
                                           TST_WorkspaceSizeInBytes);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionEnd",
                          TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            /* Delete all the sessions */
            /*-------------------------*/
            TST_Status = SST_SessionDelete(TST_SessionId1);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_SessionDelete1",
                          TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            TST_Status = SST_SessionDelete(TST_SessionId2);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_SessionDelete2",
                          TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                          TST_MasterTestName);


            TST_Status = SST_SessionDelete(TST_SessionId3);

      TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_SessionDelete3",
                          TST_ATP_BindingWritingPermissionsTestData[TST_index].TST_Name,
                          TST_MasterTestName);
  

        }/*End of TST_index "for loop" */
    }/*End of TST_HandleType "for loop" */
    return TST_Status;

  EXIT_ON_ERROR:
  {
    goto EXIT_ON_ERROR;
  }
}/* End of SST_TST_ATP_BasicDataOp */

/***************************************************************************************************************************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***************************************************************************************************************************************************************/
/*****************************************************************************
 * Function Name:                                                           
 *  SST_TST_ATP_ReadOnlyData         
 * 
 * Inputs:
 *  None
 *
 * Outputs:
 *  DxError_t - Function error return                                        
 *
 * Description:                                                                 
 * Test the creation, deletion, modification, opening, closing and usage of a password authenticator object. 
 * And demonstrate the use of transaction with authenticator objects
 *  
 * Algorithm:       
 *    1.  Create a new session by calling: SST_SessionCreate.
 *    2.  Open a transaction by calling: SST_TransactionStart.
 *    3.  Create a new password authenticator by calling: SST_AuthenticatorPasswordCreate.
 *    4.  Close the transaction by calling: SST_TransactionEnd. 
 *    5.  Open the new authenticator by calling: SST_AuthenticationOpen.
 *    6.  Open a transaction by calling: SST_TransactionStart.
 *    7.  Insert a new data object by calling: SST_InsertData.
 *    8.    Set the data to read only by calling: SST_DataReadOnlySet
 *    8.  Close the transaction by calling: SST_TransactionEnd.
 *    9.  Read and compare the inserted data by calling: SST_ReadData.
 *    10.  Open a transaction by calling: SST_TransactionStart.
 *    11.  Try to Modify the data from selected location by calling: SST_ModifyData. this should fail!!
 *    12.  Try to delete the data by calling: SST_DeleteData. this should fail!!
 *    13.  Delete the authenticator by calling: SST_AuthenticatorDelete.
 *    14.  Close the transaction by calling: SST_TransactionClose.
 *    15.  Delete the session by calling: SST_SessionDelete.
 *                            
 ********************************************************************************/    
DxError_t SST_TST_ATP_ReadOnlyData ( void )
{
    /********* TST Variables Definitions ************/
    int                 TST_index;
    char                TST_MasterTestName[TST_FUNC_STRING_LEN] = "SST_TST_ATP_ReadOnlyData";
    SSTSessionId_t      TST_SessionId; 
    SSTTxnId_t          TST_TransactionId;
    SSTHandle_t         TST_AuthHandle,TST_DataHandle;
    SSTDataType_t       TST_SimpleData = TST_READ_ONLY_DATA_OBJECT;
    DxUint32_t          TST_TempDataSize;
    SSTHandle_t       	TST_AuthDefaultHandle;
    DxError_t      		  TST_Status;

    /********* TST Variables initialization *********/

    ATP_LOG_TST_PRINT((SST_TST_MSG," ****************************************************** \n"));
    ATP_LOG_TST_PRINT((SST_TST_MSG," ** SST_TST_ATP_ReadOnlyData Tests ********************* \n"));     
    ATP_LOG_TST_PRINT((SST_TST_MSG," ****************************************************** \n \n"));

    /************************************************/
    /*********** Start of test **********************/
    /************************************************/
   

    /* As there are files that would be read only and not deletable this test works with another SST file data base */
    /* Terminate the regular SST test data base */
    /* Terminate the SST */
    /*-------------------*/
    TST_Status = SST_Terminate();

  TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_Terminate",
                      "",
                      TST_MasterTestName);


    /* Init the SST again for operations */
    /*-----------------------------------*/
    TST_Status = SST_Init((DxNvsMemoryId_t)TST_MainDBFileHandle,
               (DxNvsMemoryId_t)TST_SecondaryDBFileHandle,
                             DX_NULL,
               TST_Workspace,
               TST_WorkspaceSizeInBytes);

  TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_Init",
                        "",
                        TST_MasterTestName);


     for (TST_index =0 ; TST_index < TST_NumberOf_BasicDataOpObjectTests ; TST_index++) /*To run on all test vectors*/
     {
         ATP_LOG_TST_PRINT((SST_TST_MSG," ************************************ \n"));
         ATP_LOG_TST_PRINT((SST_TST_MSG," TEST using internal handles          \n"));
         ATP_LOG_TST_PRINT((SST_TST_MSG," ************************************ \n"));
         TST_AuthHandle.objId = SST_HANDLE_RANDOM_GENERATE; 
         TST_DataHandle.objId = SST_HANDLE_RANDOM_GENERATE;

         ATP_LOG_TST_PRINT((SST_TST_MSG,"\n",TST_ATP_DataOpTestData[TST_index].TST_Name,"\n")); 

         /* Create a session for operations */
         /*---------------------------------*/
         TST_Status = SST_SessionCreate(&TST_SessionId);

     TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_SessionCreate",
                        TST_ATP_DataOpTestData[TST_index].TST_Name,
                        TST_MasterTestName);


         /* Open a transaction for operations */
         /*-----------------------------------*/
        TST_Status = SST_TransactionStart(&TST_TransactionId);
    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionStart",
                        TST_ATP_DataOpTestData[TST_index].TST_Name,
                        TST_MasterTestName);


         /* Create a new password authenticator */
         /*-------------------------------------*/
         TST_AuthDefaultHandle.objDigest = 0;
     TST_AuthDefaultHandle.objId = SST_CLOSE_AUTH_HANDLE;
         TST_Status = SST_AuthenticatorPasswordCreate(TST_TransactionId,
                                                     TST_SessionId,
                                                     TST_ATP_DataOpTestData[TST_index].TST_Password,
                                                     TST_ATP_DataOpTestData[TST_index].TST_PasswordSizeInBytes,
                                                     TST_ATP_DataOpTestData[TST_index].TST_MaxNumOfTrials,
                                                     TST_AuthDefaultHandle,
                                                     &TST_AuthHandle,
                                                     TST_Workspace,
                                                     TST_WorkspaceSizeInBytes);

    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticatorPasswordCreate",
                        TST_ATP_DataOpTestData[TST_index].TST_Name,
                        TST_MasterTestName);


         /* Close the transaction */
         /*-----------------------*/
         TST_Status = SST_TransactionEnd(TST_TransactionId,
                       DX_TRUE,/*reflash sensitive*/
                       TST_Workspace,
                                         TST_WorkspaceSizeInBytes);

    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionEnd",
                        TST_ATP_DataOpTestData[TST_index].TST_Name,
                        TST_MasterTestName);


         /* Open the password authentication */
         /*----------------------------------*/
         TST_Status = SST_AuthenticationOpen(TST_SessionId,
                                            TST_AuthHandle,
                                            TST_ATP_DataOpTestData[TST_index].TST_Password,
                                            TST_ATP_DataOpTestData[TST_index].TST_PasswordSizeInBytes,
                                            TST_NO_CHALLENGE,
                                            TST_Workspace,
                                            TST_WorkspaceSizeInBytes);

    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticationOpen - Passeord",
                        TST_ATP_DataOpTestData[TST_index].TST_Name,
                        TST_MasterTestName);


    /* Open a transaction for operations */
         /*-----------------------------------*/
         TST_Status = SST_TransactionStart(&TST_TransactionId);

     TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionStart",
                        TST_ATP_DataOpTestData[TST_index].TST_Name,
                        TST_MasterTestName);


         TST_DataHandle.objId = TST_index;

         /* Insert a new Data object */
         /*--------------------------*/
         TST_Status = SST_DataInsert(TST_TransactionId,
                                    TST_SessionId, 
                                    TST_AuthHandle,       
                                    TST_ATP_DataOpTestData[TST_index].TST_Data,
                                    TST_ATP_DataOpTestData[TST_index].TST_DataSize,
                                    DX_TRUE,
                                    TST_SimpleData,                
                                    &TST_DataHandle,
                                    TST_Workspace,
                                    TST_WorkspaceSizeInBytes);

    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_DataInsert",
                        TST_ATP_DataOpTestData[TST_index].TST_Name,
                        TST_MasterTestName);


         /* Set the data to Read Only */
         /*---------------------------*/
         TST_Status = SST_DataReadOnlySet(TST_TransactionId,
                                         TST_SessionId, 
                                         TST_DataHandle,
                                         TST_Workspace,
                                         TST_WorkspaceSizeInBytes);

    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_DataReadOnlySet",
                        TST_ATP_DataOpTestData[TST_index].TST_Name,
                        TST_MasterTestName);



         /* Close the transaction */
         /*-----------------------*/
         TST_Status = SST_TransactionEnd(TST_TransactionId,
                     DX_TRUE,/*reflash sensitive*/
                     TST_Workspace,
                                        TST_WorkspaceSizeInBytes);

    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionEnd",
                        TST_ATP_DataOpTestData[TST_index].TST_Name,
                        TST_MasterTestName);


         /* Read the data */
         /*---------------*/
         TST_TempDataSize = TST_ATP_DataOpTestData[TST_index].TST_DataSize;

         TST_Status = SST_DataRead(TST_SessionId,
                                  TST_DataHandle,
                                  TST_ATP_DataOpTestData[TST_index].TST_DataReadOffset,
                                  TST_ATP_DataOpTestData[TST_index].TST_OutputData,
                                  &TST_TempDataSize,
                                  TST_Workspace,
                                  TST_WorkspaceSizeInBytes);

    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_DataRead",
                        TST_ATP_DataOpTestData[TST_index].TST_Name,
                        TST_MasterTestName);


         /* Compare the retrieved data to the original */
    /*--------------------------------------------*/
       if(!DX_VOS_MemCmp(TST_ATP_DataOpTestData[TST_index].TST_Data,
              TST_ATP_DataOpTestData[TST_index].TST_OutputData,
              TST_ATP_DataOpTestData[TST_index].TST_DataSize))
                           TST_Status = TESTPASS; /*Data Buffers are the same*/
                           else
                           TST_Status = TESTFAIL; /*Data Buffers are different*/
      
    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"DX_VOS_MemCmp - Original data",
                        TST_ATP_DataOpTestData[TST_index].TST_Name,
                        TST_MasterTestName);




         /* Open a transaction for operations */
         /*-----------------------------------*/
         TST_Status = SST_TransactionStart(&TST_TransactionId);

    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionStart",
                        TST_ATP_DataOpTestData[TST_index].TST_Name,
                        TST_MasterTestName);


         /* Try to Modify the data from selected location - this operation should fail!! */
         /*------------------------------------------------------------------------------*/
         TST_Status = SST_DataModify(TST_TransactionId,
                                    TST_SessionId,                           
                                    TST_DataHandle,  
                                    TST_ATP_DataOpTestData[TST_index].TST_DataToWrite,
                                    TST_ATP_DataOpTestData[TST_index].TST_DataToWriteSize,
                                    TST_ATP_DataOpTestData[TST_index].TST_DataToWriteOffset,
                                    TST_Workspace,
                                    TST_WorkspaceSizeInBytes);

     TST_Status = TST_CheckExpectedErrorReturn(TST_Status,
                                                       SST_RC_ERROR_ACCESS_DENIED,
                                                       "SST_DataModify",
                                TST_ATP_DataOpTestData[TST_index].TST_Name,
                                TST_MasterTestName);
    #if SST_TST_EXIT_ON_ERROR
      if (TST_Status != TESTPASS)
        goto EXIT_ON_ERROR;
    #endif

         /* Try to Delete the data  - This operation should fail!! */
         /*--------------------------------------------------------*/
          TST_Status = SST_DataDelete(TST_TransactionId,
                                     TST_SessionId,  
                                     TST_DataHandle,
                                     TST_Workspace,
                                     TST_WorkspaceSizeInBytes);

         TST_Status = TST_CheckExpectedErrorReturn(TST_Status,
                                                      SST_RC_ERROR_ACCESS_DENIED,
                                                      "SST_DataDelete",
                                TST_ATP_DataOpTestData[TST_index].TST_Name,
                                TST_MasterTestName);

    #if SST_TST_EXIT_ON_ERROR
      if (TST_Status != TESTPASS)
        goto EXIT_ON_ERROR;
    #endif

         /* Delete the authenticator - this should fail as the authenticator is connected to data */
         /*---------------------------------------------------------------------------------------*/
         TST_Status = SST_AuthenticatorDelete(TST_TransactionId,
                                             TST_AuthHandle);

     TST_Status = TST_CheckExpectedErrorReturn(TST_Status,
                                                       SST_RC_ERROR_ACCESS_DENIED,
                                                       "SST_AuthenticatorDelete",
                                 TST_ATP_DataOpTestData[TST_index].TST_Name,
                                 TST_MasterTestName);
    #if SST_TST_EXIT_ON_ERROR
      if (TST_Status != TESTPASS)
        goto EXIT_ON_ERROR;
    #endif

         /* Close the transaction */
         /*-----------------------*/
         TST_Status = SST_TransactionEnd(TST_TransactionId,
                     DX_TRUE,/*reflash sensitive*/
                     TST_Workspace,
                                        TST_WorkspaceSizeInBytes);

       TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionEnd",
                        TST_ATP_DataOpTestData[TST_index].TST_Name,
                        TST_MasterTestName);


         /* Delete the session */
         /*--------------------*/
         TST_Status = SST_SessionDelete(TST_SessionId);

     TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_SessionDelete",
                        TST_ATP_DataOpTestData[TST_index].TST_Name,
                        TST_MasterTestName);


     }/*End of TST_index "for loop" */

    /* Test Cleanup: */
    /* Revert back to the original test data base */
    /* Terminate the SST */
    /*-------------------*/
    TST_Status = SST_Terminate();

  TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_Terminate",
                      TST_ATP_DataOpTestData[TST_index].TST_Name,
                      TST_MasterTestName);


    /* Close and delete the database */
    /*------------------------------------*/
    /* Close the file*/
#ifndef DX_NVS_RAM
  TST_Status = DX_VOS_FClose(TST_MainDBFileHandle);

    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"DX_VOS_FClose",
                      TST_ATP_DataOpTestData[TST_index].TST_Name,
                      TST_MasterTestName);

#endif

    /* Delete the file */
#ifndef DX_NVS_RAM
  TST_Status = DX_VOS_FDelete(SST_DB_FILENAME);

    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"DX_VOS_FDelete",
                      TST_ATP_DataOpTestData[TST_index].TST_Name,
                      TST_MasterTestName);


#endif


#ifdef DX_NVS_RAM

         DX_VOS_MemSetZero(TST_MainDBPool,NVS_MEMORY_DATA_POOL_SIZE_IN_BYTES);
         DX_VOS_MemSetZero(TST_SecondaryDBPool,NVS_MEMORY_DATA_POOL_SIZE_IN_BYTES);
         
#endif
    /* reopen the file for the tests to continue */
#ifndef DX_NVS_RAM
  TST_Status = DX_VOS_FOpen(&TST_MainDBFileHandle,SST_DB_FILENAME ,"w+b");

    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"DX_VOS_FOpen",
                      TST_ATP_DataOpTestData[TST_index].TST_Name,
                      TST_MasterTestName);


#endif
    /* Init the SST again for operations */
    /*-----------------------------------*/
  TST_Status = SST_Init((DxNvsMemoryId_t)TST_MainDBFileHandle,
             (DxNvsMemoryId_t)TST_SecondaryDBFileHandle,
                         DX_NULL,
             TST_Workspace,
             TST_WorkspaceSizeInBytes);

  TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_Init",
                        TST_ATP_DataOpTestData[TST_index].TST_Name,
                        TST_MasterTestName);

        

    return TST_Status;

  EXIT_ON_ERROR:
  {
    return DX_FAILURE;
  }
}/* End of SST_TST_ATP_ReadOnlyData */

/*****************************************************************************
* Function Name:                                                           
*  SST_TST_ATP_DataReplace         
* 
* Inputs:
*  None
*
* Outputs:
*  DxError_t - Function error return
*
* Description:
* Test data replace operation
*  
* Algorithm:
*    1. Create a new session by calling SST_SessionCreate.
*    2. Open a transaction by calling SST_TransactionStart.
*    3. Create a new password authenticator (with closed master) by calling: SST_AuthenticatorPasswordCreate.
*    4. Close the transaction by calling: SST_TransactionEnd. 
*    5. Open the new authenticator by calling: SST_AuthenticationOpen.
*    6. Open a transaction by calling: SST_TransactionStart.
*    7. Insert a new data object by calling SST_InsertData.
*    8. Replace the content of the data by calling SST_DataReplace.
*    9. Verify the data was actually replaced by calling SST_DataRead and comparing the data.
*   10. Delete the data by calling SST_DataDelete.
*   11.  Delete the authenticator by calling: SST_AuthenticatorDelete.
*   12. Close the transaction by calling SST_TransactionEnd.
*   13. Delete the session by calling SST_SessionDelete.
********************************************************************************/
DxError_t SST_TST_ATP_DataReplace ( void )
{
   SSTSessionId_t   TST_SessionId;
   int				TST_index = 0;
   DxChar           * TestName = "ATP test SST Data Replace";
   SSTTxnId_t       TST_TransactionId;
   SSTHandle_t      TST_AuthDefaultHandle;
   SSTHandle_t      TST_DataHandle;
   SSTDataType_t    TST_SimpleData = TST_DATA_OBJECT;
   DxUint32_t       TST_DataSize;
   DxError_t        TST_Status;
   SSTHandle_t      TST_AuthHandle;


   DX_VOS_MemSet6(TST_DATA_BUFF1,'1',TST_BUFF1_SIZE_IN_BYTES);
   DX_VOS_MemSet6(TST_DATA_BUFF2,'2',TST_BUFF2_SIZE_IN_BYTES);
   DX_VOS_MemSet6(TST_DATA_BUFF3,'0',TST_BUFF2_SIZE_IN_BYTES);
   TST_DataHandle.objId = SST_HANDLE_RANDOM_GENERATE;

   /* Create a session for operations */
   /*---------------------------------*/
   TST_Status = SST_SessionCreate(&TST_SessionId);

   TST_ATP_CHECK_ERROR_RETURN(TST_Status,
               "SST_SessionCreate",
                              TestName,
                              TestName);


   /* Open a transaction for operations */
   /*-----------------------------------*/
   TST_Status = SST_TransactionStart(&TST_TransactionId);

   TST_ATP_CHECK_ERROR_RETURN(TST_Status,
               "SST_TransactionStart",
                              TestName,
                              TestName);


	/* Create a new password authenticator */
    /*-------------------------------------*/
    TST_AuthDefaultHandle.objDigest = 0;
	TST_AuthDefaultHandle.objId = SST_CLOSE_AUTH_HANDLE;
    TST_Status = SST_AuthenticatorPasswordCreate(	TST_TransactionId,
													TST_SessionId,
													TST_ATP_DataOpTestData[TST_index].TST_Password,
													TST_ATP_DataOpTestData[TST_index].TST_PasswordSizeInBytes,
													TST_ATP_DataOpTestData[TST_index].TST_MaxNumOfTrials,
													TST_AuthDefaultHandle,
													&TST_AuthHandle,
													TST_Workspace,
													TST_WorkspaceSizeInBytes);

	TST_ATP_CHECK_ERROR_RETURN(	TST_Status,"SST_AuthenticatorPasswordCreate",
								TestName,
								TestName);


    /* Close the transaction */
    /*-----------------------*/
    TST_Status = SST_TransactionEnd(TST_TransactionId,
									DX_TRUE,/*reflash sensitive*/
									TST_Workspace,
									TST_WorkspaceSizeInBytes);

	TST_ATP_CHECK_ERROR_RETURN(	TST_Status,"SST_TransactionEnd",
								TestName,
								TestName);


    /* Open the password authentication */
    /*----------------------------------*/
    TST_Status = SST_AuthenticationOpen(	TST_SessionId,
											TST_AuthHandle,
											TST_ATP_DataOpTestData[TST_index].TST_Password,
											TST_ATP_DataOpTestData[TST_index].TST_PasswordSizeInBytes,
											TST_NO_CHALLENGE,
											TST_Workspace,
											TST_WorkspaceSizeInBytes);

	TST_ATP_CHECK_ERROR_RETURN(	TST_Status,"SST_AuthenticationOpen - Passeord",
								TestName,
								TestName);


	/* Open a transaction for operations */
    /*-----------------------------------*/
    TST_Status = SST_TransactionStart(&TST_TransactionId);

	TST_ATP_CHECK_ERROR_RETURN(	TST_Status,"SST_TransactionStart",
								TestName,
								TestName);

   /* Insert a new Data object */
   /*--------------------------*/
   TST_Status = SST_DataInsert(	TST_TransactionId,
								TST_SessionId, 
								TST_AuthHandle,
								TST_DATA_BUFF1,
								sizeof(TST_DATA_BUFF1),
								DX_TRUE,
								TST_SimpleData,
								&TST_DataHandle,
								TST_Workspace,
								TST_WorkspaceSizeInBytes);

   TST_ATP_CHECK_ERROR_RETURN(TST_Status,
               "SST_DataInsert",
                              TestName,
                              TestName);


   /*Replace the content of the data by calling SST_DataReplace*/
   /*----------------------------------------------------------*/
   TST_Status = SST_DataReplace(TST_TransactionId,
                               TST_SessionId,
                               TST_DataHandle,
                               TST_DATA_BUFF2,
                               sizeof(TST_DATA_BUFF2),
                               DX_TRUE,
                               TST_Workspace,
                               TST_WorkspaceSizeInBytes);

   TST_ATP_CHECK_ERROR_RETURN(TST_Status,
               "SST_DataReplace",
                              TestName,
                              TestName);


   /*Verify the data was actually replaced by calling SST_DataRead and comparing the data*/
   /*------------------------------------------------------------------------------------*/
   TST_DataSize = sizeof(TST_DATA_BUFF3);
   TST_Status = SST_DataRead(TST_SessionId,
                            TST_DataHandle,
                            0,
                           TST_DATA_BUFF3,
                           &TST_DataSize,
                           TST_Workspace,
                           TST_WorkspaceSizeInBytes);
   TST_ATP_CHECK_ERROR_RETURN(TST_Status,
               "SST_DataRead",
                              TestName,
                              TestName);


   if(TST_DataSize == TST_BUFF2_SIZE_IN_BYTES)
     TST_Status = TESTPASS;
   else
     TST_Status = TESTFAIL;

   TST_ATP_CHECK_ERROR_RETURN(TST_Status,
               "Data size",
                              TestName,
                              TestName);


   if(!DX_VOS_MemCmp(TST_DATA_BUFF3,TST_DATA_BUFF2,TST_DataSize))
     TST_Status = TESTPASS;
   else
    TST_Status = TESTFAIL;


   TST_ATP_CHECK_ERROR_RETURN(TST_Status,
               "DX_VOS_MemCmp",
                              TestName,
                              TestName);


   /*Delete the data by calling SST_DataDelete*/
   /*-----------------------------------------*/
   TST_Status = SST_DataDelete(TST_TransactionId,
                              TST_SessionId,
                              TST_DataHandle,
                              TST_Workspace,
                              TST_WorkspaceSizeInBytes);

   TST_ATP_CHECK_ERROR_RETURN(TST_Status,
               "SST_DataDelete",
                              TestName,
                              TestName);

	/* Delete the authenticator */
	/*--------------------------*/
   TST_Status = SST_AuthenticatorDelete(	TST_TransactionId,
											TST_AuthHandle);

	TST_ATP_CHECK_ERROR_RETURN(	TST_Status,"SST_AuthenticatorDelete - Master authenticator",
								TestName,
								TestName);

   /*Close the transaction by calling SST_TransactionEnd*/
   /*---------------------------------------------------*/
   TST_Status = SST_TransactionEnd(TST_TransactionId,
                                 DX_TRUE,
                                 TST_Workspace,
                                 TST_WorkspaceSizeInBytes);
   TST_ATP_CHECK_ERROR_RETURN(TST_Status,
               "SST_TransactionEnd",
                              TestName,
                              TestName);


/*Delete the session by calling SST_SessionDelete*/
/*---------------------------------- ------------*/
   TST_Status = SST_SessionDelete(TST_SessionId);
   TST_ATP_CHECK_ERROR_RETURN(TST_Status,
               "SST_SessionDelete",
                              TestName,
                              TestName);

   return TST_Status;

   EXIT_ON_ERROR:
  {
    return DX_FAILURE;
  }
}

/***************************************************************************************************************************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR********************************************************/
/***--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR--TESTS SEPARATOR******************************************************/
/***************************************************************************************************************************************************************/

#endif


